/* sccsid[] = "@(#) $Id: //bas/46C/src/proj/crfc/crfcconn.cpp#2 $ SAP" */   
//
// SAP RFC Classes C++ library.
// Copyright (C) 1996 SAP America, Inc.
// All rights reserved. 

////////////////////////////////////////////////////////////////////////////////
//  File crfcconn.cpp
#include <ctype.h>
#include <stdio.h>    

#if defined(SAPonRS6000)
#pragma priority(100)
#endif

#include "sapitab.h"
#include "crfcconn.h"
#include "crfcclnt.h"
#include "crfcsimp.h"
#include "crfccomp.h"
#include "crfcstrc.h"

       
extern CRfcTrace RfcClassTrace;

#if defined(_WIN32) || defined(SAPonNT)
CSAPCriticalSection ConnectLock;  // needed for synchronizing rfc open operations
                                  // as required by SAP gateway before 3.1H
#endif


CRfcConnection::CRfcConnection (void)
                     :m_hRFC(RFC_HANDLE_NULL),
                     m_nTraceLevel (0),
                     m_bSelfCreated (FALSE),
                     m_pSystemInfoFunc(NULL),
		     m_codePage((CSTR)NULL),
                     m_SysInfo()

{
    RfcClassTrace.Trace(CRFCCONN_DEFAULT_CONSTRUCTOR);
}

CRfcConnection::CRfcConnection(const RFC_USER_INFO &userInfo, 
                                    // user logon information 
                               CSTR  destination)
                                    // R/3 system destination name, 
                                    // which is defined in either
                                    // SIDEINFO or SAPRFC.INI file
                :m_hRFC(RFC_HANDLE_NULL),
                m_nTraceLevel(0),
                m_pSystemInfoFunc(NULL),
		m_codePage((CSTR)NULL),
                m_SysInfo()
{

    
    m_UserInfo.rstrClient   = userInfo.rstrClient;
    m_UserInfo.rstrUserName = userInfo.rstrUserName;
    m_UserInfo.rstrPassword = userInfo.rstrPassword;
    m_UserInfo.rstrLanguage = userInfo.rstrLanguage;

    m_ConnectInfo.rstrDestination = destination;
    m_ConnectInfo.rfcMode         = RFC_MODE_R3ONLY;   //default mode
	m_ConnectInfo.nSAPGUI		  = 0;

    //Have not created handle yet
    m_bSelfCreated = FALSE ;
}



CRfcConnection::~CRfcConnection (void)
{

    //need to close connection
    if (m_hRFC != RFC_HANDLE_NULL)
    {
       Close();
    }
    
    if(m_pSystemInfoFunc != NULL) 
        delete m_pSystemInfoFunc;

    if(m_SysInfo.GetSize() > 0)
    {
        for(int i=0; i < m_SysInfo.GetSize(); i++)
        {
            delete m_SysInfo[i];
        }
    }
}

// Close connection and send a message  
void CRfcConnection::Abort (char* Message)
{
    RfcClassTrace.TraceTime();
    RfcClassTrace.Trace(CRFCCONN_ABORT);
    RfcClassTrace.Trace(Message);

    assert (m_hRFC != RFC_HANDLE_NULL) ;

    if (m_hRFC == RFC_HANDLE_NULL)
    {
        return;
    }
    
    ::RfcAbort (m_hRFC, Message) ;
    m_hRFC = RFC_HANDLE_NULL;

}

// Accept an incoming connection, console version
void CRfcConnection::Accept (char** argv)
{
//  Save the command line argument for multithreading.  It is ugly.  But it is easy.
#if defined(CRFCwithTHREADS)
	m_argv = argv;
#endif
    CRFC_ERROR_INFO Err;

    //Close current connection
    Close () ;

#if defined(_WIN32) || defined(SAPonNT)
	ConnectLock.Lock();
#endif

    m_hRFC = ::RfcAccept (argv);

#if defined(_WIN32) || defined(SAPonNT)
	Sleep(20);
	ConnectLock.Unlock();
#endif

    if (m_hRFC == RFC_HANDLE_NULL)
    {
        RfcClassTrace.TraceTime();
        RfcClassTrace.Trace(CRFCCONN_RFCACCEPT_FAIL);
        RfcLastErrorEx (&Err) ;
		RfcLastError (&Err) ;
        throw Err ;
    }

    RfcClassTrace.TraceTime();
    RfcClassTrace.Trace(CRFCCONN_ACCEPT);
    
	if (m_nTraceLevel != 0)
		::RfcSetTrace(m_hRFC, m_nTraceLevel);  //  Set R/3 system trace level
	if (m_codePage != (CSTR)NULL)
		::RfcSetCodePage(m_hRFC, (char *)(CSTR)m_codePage);

    m_bSelfCreated = TRUE;
}


void CRfcConnection::Clear()
{

    Close () ;

    //Clear all user information 
    m_UserInfo.rstrClient   = NULL;
    m_UserInfo.rstrUserName = NULL;
    m_UserInfo.rstrPassword = NULL;
    m_UserInfo.rstrLanguage = NULL;

    //Clear connection information
    m_ConnectInfo.rfcMode            = RFC_MODE_R3ONLY;
    m_ConnectInfo.nSAPGUI            = 0;
	m_ConnectInfo.cRfcServerType     = '3';
	m_ConnectInfo.rstrProgramID      = NULL;
    m_ConnectInfo.rstrR3release      = RELEASEUNKNOWN;
    m_ConnectInfo.bGetR3SystemInfo   = FALSE;
    m_ConnectInfo.rstrDestination    = NULL;
    m_ConnectInfo.rstrHostName       = NULL;
    m_ConnectInfo.rstrGatewayHost    = NULL;
    m_ConnectInfo.rstrGatewayService = NULL;
    m_ConnectInfo.nSystemNo          = 0 ;

    //Clear connection information for VERSION 3
    //fields
    m_ConnectInfo.rstrSystemName     = NULL;
    m_ConnectInfo.rstrMsgServer      = NULL;
    m_ConnectInfo.rstrGroupName      = NULL;

    m_nTraceLevel = 0 ;
}


// Close this connection;
void CRfcConnection::Close(void)
{

    if( m_hRFC == RFC_HANDLE_NULL ||
         !m_bSelfCreated)
         return;

     ::RfcClose (m_hRFC);

     RfcClassTrace.TraceTime();
     RfcClassTrace.Trace(CRFCCONN_CLOSE);

     m_hRFC = RFC_HANDLE_NULL;
}


void CRfcConnection::ConnArgv(char** argv)
{
    RFC_OPTIONS           Options;
    RFC_CONNOPT_R3ONLY    ConnoptR3Only;
    RFC_CONNOPT_CPIC      ConnoptCPIC;
    RFC_CONNOPT_VERSION_3 ConnoptR3Version3;


    Clear() ;

    memset((void*)&Options, 0, sizeof(RFC_OPTIONS));
    memset((void*)&ConnoptR3Only, 0, sizeof(RFC_CONNOPT_R3ONLY));
    memset((void*)&ConnoptR3Version3, 0, sizeof(RFC_CONNOPT_VERSION_3));

	::RfcConnArgv3 ( (char**)argv,
                     &Options, 
                     &ConnoptCPIC,
                     &ConnoptR3Only,
                     &ConnoptR3Version3);

    //Setup user sign on infomation
    m_UserInfo.rstrClient           = Options.client;
    m_UserInfo.rstrUserName         = Options.user;
    m_UserInfo.rstrPassword         = Options.password;
    m_UserInfo.rstrLanguage         = Options.language;
    m_ConnectInfo.rstrDestination   = Options.destination;
	m_ConnectInfo.nSAPGUI			= 0;
    m_nTraceLevel                   = Options.trace;

	RFC_HANDLE rfc_handle = GetHandle();
	if (rfc_handle != RFC_HANDLE_NULL && m_nTraceLevel != 0)
	    ::RfcSetTrace(rfc_handle, m_nTraceLevel);  //  Set R/3 system trace level

    //Depend on differnt RFC_MODE, only some connection information
    //need to be setup. 
    switch (Options.mode)
    {
    case RFC_MODE_CPIC:
        m_ConnectInfo.rstrGatewayHost    = ConnoptCPIC.gateway_host;     
        m_ConnectInfo.rstrGatewayService = ConnoptCPIC.gateway_service; 
        break;

    case RFC_MODE_R3ONLY:
        m_ConnectInfo.rstrHostName       = ConnoptR3Only.hostname;
        m_ConnectInfo.nSystemNo          = ConnoptR3Only.sysnr ;
        m_ConnectInfo.rstrGatewayHost    = ConnoptR3Only.gateway_host;     
        m_ConnectInfo.rstrGatewayService = ConnoptR3Only.gateway_service;
        break; 

    case RFC_MODE_VERSION_3:
        m_ConnectInfo.nSAPGUI            = ConnoptR3Version3.use_sapgui;

        if (ConnoptR3Version3.use_load_balancing != 0)
        {
            m_ConnectInfo.rstrSystemName = ConnoptR3Version3.lb_system_name ;
            m_ConnectInfo.rstrGroupName  = ConnoptR3Version3.lb_group ;
            m_ConnectInfo.rstrMsgServer  = ConnoptR3Version3.lb_host ;
        }
        else
        {
           m_ConnectInfo.rstrHostName    = ConnoptR3Version3.hostname;
           m_ConnectInfo.nSystemNo       = ConnoptR3Version3.sysnr ;
        }
        break;
    case RFC_MODE_PARAMETER:
        break;
    default:
        assert (FALSE);
    }
}


// Open an RFC connection with load balancing 
void CRfcConnection::Connect(void)
{
  //  RFC_CONNOPT_VERSION_3 ConnoptR3Version3;
	//  No longer needed for RfcOpenEx

    //do nothing if connection already established
    if( m_hRFC != RFC_HANDLE_NULL )
        return;

	// do not allow rfc with SAPGUI
	m_ConnectInfo.nSAPGUI		  = 0;

	DoOpen();   //  Use RfcOpenEx

//  Some codes have been deleted here when RfcOpenEx is uesed to replace RfcOpen
}

//  The old implementation of DoOpen for RfcOpen have been deleted here when RfcOpenEx is uesed to replace RfcOpen

//Pack parameters and do the actual open
//Throws exception when open failed (using RfcOpenEx)
void CRfcConnection::DoOpen (void)
{
    CRFC_ERROR_INFO Err;

	CRfcString connect_param;
	ConnectParamFromConnectInfo(connect_param);

#if defined(_WIN32) || defined(SAPonNT)
	ConnectLock.Lock();
#endif

    m_hRFC = ::RfcOpenEx ((char *)(CSTR)connect_param, &Err) ;

#if defined(_WIN32) || defined(SAPonNT)
	Sleep(20);
	ConnectLock.Unlock();
#endif

    if (m_hRFC == RFC_HANDLE_NULL)
    {
        RfcClassTrace.TraceTime();
        RfcClassTrace.Trace(CRFCCONN_DOOPEN_FAILURE);
        RfcClassTrace.Trace(m_ConnectInfo.rstrDestination);
        RfcClassTrace.Trace(m_UserInfo.rstrClient);
        RfcClassTrace.Trace(m_UserInfo.rstrUserName);
        RfcLastErrorEx (&Err) ;
		RfcLastError (&Err) ;
        RfcClassTrace.TraceRFCErrorInfo((RFC_ERROR_INFO_EX) Err);
        throw Err ;
    }

    if( GetSafeHandle() == RFC_HANDLE_NULL )
    {
        // the Err obtained could be empty in this case
        RfcLastErrorEx (&Err) ;
		RfcLastError (&Err) ;
        throw Err ;
    }
	
	if (m_nTraceLevel != 0)
		::RfcSetTrace(m_hRFC, m_nTraceLevel);  //  Set R/3 system trace level
	if (m_codePage != (CSTR)NULL)
		::RfcSetCodePage(m_hRFC, (char *)(CSTR)m_codePage);  //  Set codepage

    RfcClassTrace.TraceTime();
    RfcClassTrace.Trace(CRFCCONN_DOOPEN);
    RfcClassTrace.Trace("Opening connection to: ");
    RfcClassTrace.Trace(m_ConnectInfo.rstrDestination);
    RfcClassTrace.Trace(m_UserInfo.rstrClient);
    RfcClassTrace.Trace(m_UserInfo.rstrUserName);

    //A self created handle
    m_bSelfCreated = TRUE ;

    // check and see if we need to call RfcGetAttributes to get
    //  release info
    if(m_ConnectInfo.rstrR3release != RELEASEUNKNOWN &&
	   m_ConnectInfo.rstrR3release != "")
    {
        m_ConnectInfo.rstrR3release.ToUpper();
        return;
    }

    GetR3Release();

    if(m_ConnectInfo.bGetR3SystemInfo && (m_pSystemInfoFunc == NULL))
        GetR3SystemInfo();

}

void CRfcConnection::SetUserInfo(const RFC_USER_INFO &UserInfo )
{
    
    //if there an existing connection with the old user info,
    //close that one first
    Close();

    m_UserInfo.rstrClient   = UserInfo.rstrClient  ;  
    m_UserInfo.rstrUserName = UserInfo.rstrUserName;
    m_UserInfo.rstrPassword = UserInfo.rstrPassword;
    m_UserInfo.rstrLanguage = UserInfo.rstrLanguage;
}

void CRfcConnection::SetConnectInfo(const RFC_CONNECT_INFO &ConnectInfo )
{
    //if there an existing connection with the old connect info,
    //close that one first
    Close();

	// do not allow rfc with SAPGUI
    m_ConnectInfo.rfcMode = ConnectInfo.rfcMode;
    m_ConnectInfo.nSAPGUI = ConnectInfo.nSAPGUI;
	//m_ConnectInfo.bSAPGUI = FALSE;
	m_ConnectInfo.cRfcServerType = ConnectInfo.cRfcServerType;
	m_ConnectInfo.rstrProgramID = ConnectInfo.rstrProgramID;
    m_ConnectInfo.rstrR3release = ConnectInfo.rstrR3release;
    m_ConnectInfo.bGetR3SystemInfo = ConnectInfo.bGetR3SystemInfo;
    m_ConnectInfo.rstrDestination    = ConnectInfo.rstrDestination;    
    m_ConnectInfo.rstrHostName       = ConnectInfo.rstrHostName;       
    m_ConnectInfo.rstrGatewayHost    = ConnectInfo.rstrGatewayHost;    
    m_ConnectInfo.rstrGatewayService = ConnectInfo.rstrGatewayService; 
    m_ConnectInfo.nSystemNo          = ConnectInfo.nSystemNo;        
    m_ConnectInfo.rstrSystemName     = ConnectInfo.rstrSystemName;  
    m_ConnectInfo.rstrMsgServer      = ConnectInfo.rstrMsgServer;    
    m_ConnectInfo.rstrGroupName      = ConnectInfo.rstrGroupName;    
}

// Make sure a connection to the target system is
// still alive and user sigon information is validated
// When the RFC handle is not "safe", return RFC_HANDLE_NULL
RFC_HANDLE CRfcConnection::GetSafeHandle(void)
{
   char* Exception;
   RFC_RC rc;
   CRfcClientFunc* pRFCPING;

   if (m_hRFC != RFC_HANDLE_NULL) 
   {
       // Make sure the connection is still alive
       try 
       {
          //First need to create an RFCPING object
          pRFCPING = new CRfcClientFunc(this, "RFCPING");
          if( pRFCPING == NULL )
          {
              RfcClassTrace.Trace(MEMORY);
          }

          //Ping the target system, and verify use
          //sigon information
          rc = pRFCPING->CallReceive (Exception);

          //Before existing, destroy the RFCPING object
          delete pRFCPING;
		  pRFCPING = NULL;

          if( rc != RFC_OK )
          {
              if( rc == RFC_CLOSED ) //connection closed by the other end
                  m_hRFC = RFC_HANDLE_NULL;

			 if (pRFCPING !=  NULL)
                 delete pRFCPING;
             return RFC_HANDLE_NULL;
          }
       }
       catch (RFC_ERROR_INFO_EX err)
       {
		   DumpRFCErrorInfo(err);
          //Before existing, destroy the RFCPING object
           delete pRFCPING;

          //m_hRFC = RFC_HANDLE_NULL ;
           return RFC_HANDLE_NULL;
       }
	   catch (char * pException)
	   {
		   printf("CRfcConnection::GetSafeHandle, %s\n", pException);
		   delete pRFCPING;
		   return RFC_HANDLE_NULL;
	   }
   }

   return m_hRFC ;
}

// Open an RFC connection
void CRfcConnection::Open(void)
{
	if( m_hRFC != RFC_HANDLE_NULL )
        return;

	DoOpen();

#if 0  //  With RfcOpenEx, implementation of Open is very different
    RFC_CONNOPT_R3ONLY    ConnoptR3Only;
    RFC_CONNOPT_CPIC      ConnoptCPIC;
    RFC_CONNOPT_VERSION_3 ConnoptR3Version3;

    //do nothing if connection already established
    if( m_hRFC != RFC_HANDLE_NULL )
        return;

    memset((void*)&ConnoptR3Only, 0, sizeof(RFC_CONNOPT_R3ONLY));
    memset((void*)&ConnoptCPIC, 0, sizeof(RFC_CONNOPT_CPIC));
    memset((void*)&ConnoptR3Version3, 0, sizeof(RFC_CONNOPT_VERSION_3));

    switch (m_ConnectInfo.rfcMode) 
    {
    case RFC_MODE_CPIC:
        if ((CSTR) m_ConnectInfo.rstrGatewayHost == NULL ||
            (CSTR) m_ConnectInfo.rstrGatewayService  == NULL)
        {
            //No sufficient system info provided, indicate
            //to DoOpen that sideinfo is to be used
            DoOpen(NULL);
        }
        else
        {
            ConnoptCPIC.gateway_host    = (char*)(CSTR) m_ConnectInfo.rstrGatewayHost     ;     
            ConnoptCPIC.gateway_service = (char*)(CSTR) m_ConnectInfo.rstrGatewayService  ; 
            DoOpen (&ConnoptCPIC) ;
        }
        break; 

    case RFC_MODE_R3ONLY:
        if ((CSTR) m_ConnectInfo.rstrHostName == NULL ||
            (CSTR) m_ConnectInfo.rstrGatewayHost  == NULL ||
            (CSTR) m_ConnectInfo.rstrGatewayService  == NULL)
        { //use sideinfo
            DoOpen(NULL);
        }
        else
        {
            ConnoptR3Only.hostname        = (char*)(CSTR)m_ConnectInfo.rstrHostName       ;
            ConnoptR3Only.sysnr           = m_ConnectInfo.nSystemNo          ;
            ConnoptR3Only.gateway_host    = (char*)(CSTR)m_ConnectInfo.rstrGatewayHost    ;     
            ConnoptR3Only.gateway_service = (char*)(CSTR)m_ConnectInfo.rstrGatewayService ;
            DoOpen (&ConnoptR3Only);
        }
        break; 

    case RFC_MODE_VERSION_3:
//        if (  ((CSTR) m_ConnectInfo.rstrHostName == NULL ||
//               (CSTR) m_ConnectInfo.rstrGroupName  == NULL ||
//               (CSTR) m_ConnectInfo.rstrMsgServer  == NULL))
//            && (CSTR) m_ConnectInfo.rstrHostName == NULL  )
        if( (CSTR) m_ConnectInfo.rstrHostName == NULL )
        {
            //use sideinfo
            DoOpen(NULL);
        }
        else
        {
            ConnoptR3Version3.use_sapgui        = m_ConnectInfo.nSAPGUI;
            ConnoptR3Version3.lb_system_name    = NULL;
            ConnoptR3Version3.lb_group          = NULL;
            ConnoptR3Version3.lb_host           = NULL;
            ConnoptR3Version3.use_load_balancing= FALSE;
            ConnoptR3Version3.hostname          = (char*)(CSTR)m_ConnectInfo.rstrHostName;
            ConnoptR3Version3.sysnr             = m_ConnectInfo.nSystemNo ;  
            DoOpen (&ConnoptR3Version3);
        }
        break;
    case RFC_MODE_PARAMETER: //use info in saprfc.ini
        DoOpen(NULL);
        break;
    default:
        assert (FALSE);
    }
#endif  //  With RfcOpenEx, implementation of Open is very different.
}

// Open an RFC connection and verify user logon info
// Throws const char* exception if failed
void CRfcConnection::SafeOpen(void)
{
    CRFC_ERROR_INFO Err;

    Open ();
    if( GetSafeHandle() == RFC_HANDLE_NULL )
    {
        // the Err obtained could be empty in this case
        RfcLastErrorEx (&Err) ;
		RfcLastError (&Err) ;
        throw Err ;
    }
}

 
// Open an RFC connection with load balancing 
// and user logon info verification 
// Throws const char* exception if failed   
void CRfcConnection::SafeConnect (void)
{
    CRFC_ERROR_INFO Err;

    Connect ();
    if( GetSafeHandle() == RFC_HANDLE_NULL )
    {
        // the Err obtained could be empty in this case
        RfcLastErrorEx (&Err) ;
		RfcLastError (&Err) ;
        throw Err ;
    }
}




	
#ifdef _WIN32
//Accept an incoming connection,
//Windows application only
void CRfcConnection::Accept (char* CommandLine)
{
    CRFC_ERROR_INFO Err ;

    //Close current connection
    Close () ;
#if defined(_WIN32) || defined(SAPonNT)
	ConnectLock.Lock();
#endif

    m_hRFC = ::RfcAcceptExt (CommandLine);

#if defined(_WIN32) || defined(SAPonNT)
	Sleep(20);
	ConnectLock.Unlock();
#endif

    if (m_hRFC == RFC_HANDLE_NULL)
    {
        RfcClassTrace.TraceTime();
        RfcClassTrace.Trace(CRFCCONN_ACCEPTWIN32_FAILURE);
        RfcClassTrace.Trace(CommandLine);
        RfcLastErrorEx (&Err) ;
		RfcLastError (&Err) ;
        throw Err ;
    }

	
	if (m_nTraceLevel != 0)
		::RfcSetTrace(m_hRFC, m_nTraceLevel);  //  Set R/3 system trace level
	if (m_codePage != (CSTR)NULL)
		::RfcSetCodePage(m_hRFC, (char *)(CSTR)m_codePage);  //  Set codepage

    RfcClassTrace.TraceTime();
    RfcClassTrace.Trace(CRFCCONN_ACCEPTWIN32);
    RfcClassTrace.Trace(CommandLine);

    m_bSelfCreated = TRUE;
}

  
void CRfcConnection::WriteRegistry(HKEY hKeyRoot, CSTR regPath)
{
    HKEY   hKey;
    CHAR   Buf[LINE_LEN];
    DWORD  retCode;
    DWORD  dwDisposition ;
    int    nLen;


    retCode = RegCreateKeyEx (
                hKeyRoot,                // Key handle at root level.
                regPath,                 // Path name of child key.
                0,                       // Reserved.
                "",                      // class of the key
                REG_OPTION_NON_VOLATILE, // key option  
                KEY_WRITE,               // Requesting read access.
                NULL,                    // Use the default security descriptor
                &hKey,                   // Address of key to be returned.
                &dwDisposition) ;        // Disposition values

    if (retCode)
    {
        wsprintf (Buf, "Error: RegCreateKeyEx = %d", retCode);

        RfcClassTrace.TraceTime();
		RfcClassTrace.Trace(CRFCCONN_WRITEREGISTRY);
        RfcClassTrace.Trace(Buf);

        throw CRFCCONN_WRITEREGISTRY_KEYERROR; 
    }

    //Write Destination
    SetValue (hKey, DEST, REG_SZ, 
              (BYTE*)(CSTR) m_ConnectInfo.rstrDestination, 
              m_ConnectInfo.rstrDestination.GetLength() + 1);

    //Write User
    SetValue (hKey, USER, REG_SZ, 
              (BYTE*)(CSTR)m_UserInfo.rstrUserName, 
              m_UserInfo.rstrUserName.GetLength() + 1);

    //Set Encryption flag to True when save
    BOOL bPasswdEncryption = TRUE ;
    SetValue (hKey, ENCRYPTION, REG_DWORD, (BYTE*)&bPasswdEncryption, 
              sizeof(BOOL)); 

    //Write Password
    CRfcString rfcString = m_UserInfo.rstrPassword; 
    Encrypt (rfcString) ;
    SetValue (hKey, PASSWORD, REG_SZ, 
              (BYTE*)(CSTR)rfcString,
              rfcString.GetLength() + 1); 


    //Write Client
    SetValue (hKey, CLIENT, REG_SZ, 
              (BYTE*)(CSTR)m_UserInfo.rstrClient, 
              m_UserInfo.rstrClient.GetLength() + 1); 

    //Write Language
    SetValue (hKey, LANGUAGE, REG_SZ, 
              (BYTE*)(CSTR)m_UserInfo.rstrLanguage, 
              m_UserInfo.rstrLanguage.GetLength() + 1); 

    //Write Hostname if provided
    nLen = m_ConnectInfo.rstrHostName.GetLength();
    if( nLen >= 0 ) //non-null
        SetValue (hKey, HOSTNAME, REG_SZ, 
                  (BYTE*)(CSTR)m_ConnectInfo.rstrHostName, 
                   nLen + 1); 

    //Write System Number
    SetValue (hKey, SYSTEMNUM, REG_DWORD, 
              (BYTE*)&m_ConnectInfo.nSystemNo, 
              sizeof(int)); 

    //Write Gateway Host if provided
    nLen = m_ConnectInfo.rstrGatewayHost.GetLength();
    if( nLen >= 0 ) //non-null
        SetValue (hKey, GATEWAYHOST, REG_SZ, 
                  (BYTE*)(CSTR)m_ConnectInfo.rstrGatewayHost, 
                  m_ConnectInfo.rstrGatewayHost.GetLength() + 1); 

    //Write Gateway Service if provided
    nLen = m_ConnectInfo.rstrGatewayService.GetLength();
    if( nLen >= 0 ) //non-null
        SetValue (hKey, GATEWAYSERV, REG_SZ, 
                  (BYTE*)(CSTR)m_ConnectInfo.rstrGatewayService,
                  m_ConnectInfo.rstrGatewayService.GetLength() + 1); 

    //Write Group name if provided
    nLen = m_ConnectInfo.rstrGroupName.GetLength();
    if( nLen >= 0 ) //non-null
        SetValue (hKey, GROUP, REG_SZ, 
                  (BYTE*)(CSTR)m_ConnectInfo.rstrGroupName,
                  m_ConnectInfo.rstrGroupName.GetLength() + 1); 

    //Write Message Server if provided
    nLen = m_ConnectInfo.rstrMsgServer.GetLength();
    if( nLen >= 0 ) //non-null
        SetValue (hKey, MSGSERVER, REG_SZ, 
                  (BYTE*)(CSTR)m_ConnectInfo.rstrMsgServer,
                  m_ConnectInfo.rstrMsgServer.GetLength() + 1) ;
    
    //Write System Name if provided
    nLen = m_ConnectInfo.rstrSystemName.GetLength();
    if( nLen >= 0 ) //non-null
        SetValue (hKey, SYSTEMNAME, REG_SZ, 
                  (BYTE*)(CSTR)m_ConnectInfo.rstrSystemName, 
                  m_ConnectInfo.rstrSystemName.GetLength() + 1); 

    //Write SAPGUI Mode
    SetValue (hKey, USESAPGUI, REG_DWORD, 
              (BYTE*)&m_ConnectInfo.nSAPGUI,
              sizeof(RFC_INT)); 

    //Write RFC Mode
    SetValue (hKey, RFCMODE, REG_DWORD, 
              (BYTE*)&m_ConnectInfo.rfcMode, 
              sizeof(BOOL)); 
 

}


void CRfcConnection::ReadRegistry(HKEY hKeyRoot, CSTR RegPath)
{
  HKEY   hKey;
  CHAR   Buf[LINE_LEN];
  DWORD  retCode;
  BOOL   bPasswdEncryption = FALSE ;

  BYTE   bData[MAX_VALUE_LEN];

  RfcClassTrace.Trace(CRFCCONN_READREGISTRY);
  RfcClassTrace.Trace(RegPath);

  retCode = RegOpenKeyEx (
                hKeyRoot,    // Key handle at root level.
                RegPath,     // Path name of child key.
                0,           // Reserved.
                KEY_EXECUTE, // Requesting read access.
                &hKey);      // Address of key to be returned.

  if (retCode)
  {
     wsprintf (Buf, "Error: RegOpenKeyEx = %d", retCode);

     RfcClassTrace.TraceTime();
	 RfcClassTrace.Trace(CRFCCONN_READREGISTRY);
     RfcClassTrace.Trace(Buf);

     throw CRFCCONN_READREGISTRY_KEYERROR; 
  }

  //Query Destination
  if (ReadValue (hKey, DEST, bData))
     m_ConnectInfo.rstrDestination = (char*)bData; 

  //Query User
  if (ReadValue (hKey, USER, bData))
     m_UserInfo.rstrUserName = (char*)bData;

   //Query Ecryption setting
  if (ReadValue (hKey, ENCRYPTION, bData))
     bPasswdEncryption = *(BOOL*)bData; 

  //Query Password
  if (ReadValue (hKey, PASSWORD, bData))
  {
     m_UserInfo.rstrPassword = (char*)bData;
     if (bPasswdEncryption)
     { 
        Decrypt(m_UserInfo.rstrPassword);
      }
  }

  //Query Client
  if (ReadValue (hKey, CLIENT, bData))
     m_UserInfo.rstrClient = (char*)bData; 

  //Query Language
  if (ReadValue (hKey, LANGUAGE, bData))
     m_UserInfo.rstrLanguage = (char*)bData; 

  //Query Hostname
  if (ReadValue (hKey, HOSTNAME, bData))
     m_ConnectInfo.rstrHostName = (char*)bData; 

  //Query System number
  if (ReadValue (hKey, SYSTEMNUM, bData))
     m_ConnectInfo.nSystemNo = *(DWORD*)bData; 

  //Query Gateway Host
  if (ReadValue (hKey, GATEWAYHOST, bData))
     m_ConnectInfo.rstrGatewayHost = (char*)bData; 

  //Query Gateway Service
  if (ReadValue (hKey, GATEWAYSERV, bData))
     m_ConnectInfo.rstrGatewayService = (char*)bData; 

  //Query Group name
  if (ReadValue (hKey, GROUP, bData))
     m_ConnectInfo.rstrGroupName = (char*)bData; 

  //Query Message Server
  if (ReadValue (hKey, MSGSERVER, bData))
     m_ConnectInfo.rstrMsgServer = (char*)bData; 

  //Query System Name
  if (ReadValue (hKey, SYSTEMNAME, bData))
     m_ConnectInfo.rstrSystemName = (char*)bData; 

  //Query SAPGUI Mode
  if (ReadValue (hKey, USESAPGUI, bData))
     m_ConnectInfo.nSAPGUI = *((RFC_INT*)bData);

  //Write RFC Mode
  if (ReadValue (hKey, RFCMODE, bData)) 
     m_ConnectInfo.rfcMode = *((RFC_MODE*)bData); 

}

 
//Read the registry key for specified value name
//
BOOL CRfcConnection::ReadValue (
                         HKEY hKey,       // An opened registry key
                         CSTR ValueName,  // Name of the value
                         BYTE* pbData)    // Data Value
{
    DWORD  cbData      = MAX_VALUE_LEN ;
    DWORD  dwType;
    DWORD  retCode;

    // clean out data buffer first
    assert( pbData != NULL );
	if (pbData == NULL)
		throw MEMORY;
    memset( pbData, 0, strlen((const char*)pbData));

    retCode = RegQueryValueEx (
                hKey,          // Key handle returned from RegOpenKeyEx.
                ValueName,     // Name of value.
                NULL,          // Reserved, dword = NULL.
                &dwType,       // Type of data.
                pbData,        // Data buffer.
                &cbData);      // Size of data buffer.

    return (retCode == ERROR_SUCCESS) ;
}


//Set specified value name with a value using registry key
//
BOOL CRfcConnection::SetValue (
                         HKEY    hKey,        // An opened registry key
                         CSTR   lpValueName,  // Address of subkeyName
                         DWORD   dwType,      // Type of value
                         CONST BYTE*   pbData,// address of value data
                         DWORD   cbData)      // Size of value data
                         
{
    DWORD  retCode;

    retCode = RegSetValueEx (
                    hKey,              // An opened registry key
                    lpValueName,       // Address of subkeyName
                    NULL,              // Reserved
                    dwType,            // Type of value data.
                    pbData,            // adreess of value data.
                    cbData);           // Size of value data 

 
    return  (retCode == ERROR_SUCCESS) ;
}
#endif                              


void CRfcConnection::GetR3Release(void)
{
// RFC_RC          rc;
// char            *pException;

    assert(m_hRFC != RFC_HANDLE_NULL);

	if (m_hRFC == RFC_HANDLE_NULL)
	{
		throw RFC_CONNECTION_HANDLE_NULL;
	}

    ::RfcGetAttributes(m_hRFC, &m_Attributes);

    if(m_Attributes.partner_rel == (CSTR)NULL || strlen(m_Attributes.partner_rel) == 0)
    {
        // to be here, it means that the R/3 we're dealing with is of a version
        //  prior to 4.0A, so we still need to call RFC_GET_SAP_SYSTEM_PARAMETERS.
        GetR3SystemParameters();
    }
    else
    {
        // First check to see if 'partner_rel' is empty.  If it is, then use 'kernel_rel'.
        if(m_Attributes.partner_rel[0] == ' ')
        {
            m_ConnectInfo.rstrR3release = m_Attributes.kernel_rel;
            if(m_Attributes.kernel_rel[3] == ' ')
                m_ConnectInfo.rstrR3release = m_ConnectInfo.rstrR3release.Substr(0,3);
        }
        else
        {
            m_ConnectInfo.rstrR3release = m_Attributes.partner_rel;
            if(m_Attributes.partner_rel[3] == ' ')
                m_ConnectInfo.rstrR3release = m_ConnectInfo.rstrR3release.Substr(0,3);
        }
    }

}


void CRfcConnection::GetR3SystemParameters(void)
{
RFC_RC          rc;
char            *pException;

    assert(m_hRFC != RFC_HANDLE_NULL);
	if (m_hRFC == RFC_HANDLE_NULL)
	{
		throw RFC_CONNECTION_HANDLE_NULL;
	}
    CRfcClientFunc SysParam(this, "RFC_GET_SAP_SYSTEM_PARAMETERS", FALSE);

    CRfcSimpleParam* pDateformat       = new CRfcSimpleParam("DATE_FORMAT",        TYPC, 30);
    CRfcSimpleParam* pDecimalsign      = new CRfcSimpleParam("DECIMAL_SIGN",       TYPC,  1);
    CRfcSimpleParam* pLanguage         = new CRfcSimpleParam("LANGUAGE",           TYPC,  1);
    CRfcSimpleParam* psapsystemrelease = new CRfcSimpleParam("SAP_SYSTEM_RELEASE", TYPC,  4);
    CRfcSimpleParam* pUsername         = new CRfcSimpleParam("USER_NAME",          TYPC, 12);

    SysParam.AddExportParam(*pDateformat);
    SysParam.AddExportParam(*pDecimalsign);
    SysParam.AddExportParam(*pLanguage);
    SysParam.AddExportParam(*psapsystemrelease);
    SysParam.AddExportParam(*pUsername);

	try
	{
		rc = SysParam.CallReceive(pException);
	}
    catch (RFC_ERROR_INFO_EX err)
    {
		   DumpRFCErrorInfo(err);
   
		   delete pDateformat;
           delete pDecimalsign;
           delete pLanguage;
           delete psapsystemrelease;
           delete pUsername;
    }
   
    if(rc != RFC_OK)
    {
        delete pDateformat;
        delete pDecimalsign;
        delete pLanguage;
        delete psapsystemrelease;
        delete pUsername;
        RfcClassTrace.TraceTime();
        RfcClassTrace.Trace(CRFCCONN_GETR3RELEASE_FAILURE);
        throw pException;
    }

    m_ConnectInfo.rstrR3release = (CSTR)psapsystemrelease->Value();

    //clean up!
    delete pDateformat;
    delete pDecimalsign;
    delete pLanguage;
    delete psapsystemrelease;
    delete pUsername;
}

                               
void CRfcConnection::GetR3SystemInfo()
{
RFC_RC                  rc;
char                    *pException;
CRfcImpExpParam         *pParam;
CRfcSimpleParam         *pSimp;
CRfcStructParam         *pStruct;
CRfcSimpleParam         *pSI;
const RFC_FIELD_INFO    *pFieldInfo;
unsigned                nType;
int                     nLen, nDecimals;

    assert(m_hRFC != RFC_HANDLE_NULL);
	if (m_hRFC == RFC_HANDLE_NULL)
	{
		throw RFC_CONNECTION_HANDLE_NULL;
	}
    
    if(m_pSystemInfoFunc == NULL)
    {
        m_pSystemInfoFunc = new CRfcClientFunc(this, "RFC_SYSTEM_INFO", TRUE);

        try
        {
            rc = m_pSystemInfoFunc->CallReceive(pException);
        }
        catch(RFC_ERROR_INFO_EX err)
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCCONN_GETR3SYSTEMINFO_FAILURE);
            RfcClassTrace.TraceRFCErrorInfo(err);
			DumpRFCErrorInfo(err);
        }

        if(rc != RFC_OK)
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCCONN_GETR3SYSTEMINFO_FAILURE);
            throw pException;
        }

        int nExports = m_pSystemInfoFunc->GetExportCount();

        // loop over all exported parameters
        //  See comments on CRfcSystemInfo in crfcconn.h
        for(int iP=0; iP < nExports; iP++)
        {
            pParam = (CRfcImpExpParam*)m_pSystemInfoFunc->GetExportParam(iP);
            if(pParam->IsSimpleParam())
            {
                pSimp = (CRfcSimpleParam*)pParam;
                (pSimp->Value()).GetAttributes(nType, nLen, nDecimals);
                pSI = new CRfcSimpleParam(pSimp->GetParamName());
                pSI->SetAttributes(nType, nLen, nDecimals);
                pSI->Value() = pSimp->Value();
                m_SysInfo.AppendElement(*pSI);
            }
            else
            {   // Is structure parameter
                pStruct = (CRfcStructParam*)pParam;
                
                // loop over all fields
                int nFields = pStruct->GetFieldCount();
                for(int iF=0; iF < nFields; iF++)
                {
                    pFieldInfo = pStruct->GetFieldInfo(iF);
                    ((*pStruct)[iF]).GetAttributes(nType, nLen, nDecimals);
                    pSI = new CRfcSimpleParam(pFieldInfo->strName);
                    pSI->SetAttributes(nType, nLen, nDecimals);
                    pSI->Value() = (*pStruct)[iF];
                    m_SysInfo.AppendElement(*pSI);
                }
            }
        }
    }

}


void CRfcConnection::FromHandle(RFC_HANDLE RfcHandle, BOOL IsClient) 
{
    // close existing connection before using
    // new handle
    Close() ;

    m_hRFC = RfcHandle ;

    //Not a self created handle
    m_bSelfCreated = FALSE ;

    if(IsClient)
        GetR3Release();
}



void CRfcConnection::SetTraceLevel (int nTraceLevel)
{ 
    m_nTraceLevel = nTraceLevel; 
	RFC_HANDLE rfc_handle = GetHandle();
	if (rfc_handle != RFC_HANDLE_NULL)
	    ::RfcSetTrace(rfc_handle, m_nTraceLevel);  //  Set R/3 system trace level

}


const CRfcSimpleParam& CRfcConnection::R3SystemInfo(CSTR InfoName)
{
    if(m_pSystemInfoFunc == NULL)
        GetR3SystemInfo();
    return *m_SysInfo[InfoName];

}

const CRfcSimpleParam& CRfcConnection::R3SystemInfo(int nIndex)
{
    if(m_pSystemInfoFunc == NULL)
        GetR3SystemInfo();
    return *m_SysInfo[nIndex];

}

int CRfcConnection::R3SystemInfoCount(void)
{
    if(m_pSystemInfoFunc == NULL)
        GetR3SystemInfo();
    return m_SysInfo.GetSize();
}


//  Set codepage 

void CRfcConnection::SetCodePage(char * codepage)
{
	RFC_HANDLE rfc_handle = GetHandle();
	if (rfc_handle != RFC_HANDLE_NULL && codepage != NULL)
		::RfcSetCodePage(rfc_handle, codepage);

    m_codePage = codepage;
}


//
//********************************************************************************
//  This method is used to convert RFC_CONNECT_INFO and RFC_USER_INFO stored 
//  in the CRfcConnection object to the parameter list in connect_param in for 
//  format used in RfcOpenEx(char * connect_param, RFC_ERROR_INFO_EX * error_info) 
//  Memory is allocated in the calling function for this method.
//*********************************************************************************
//
void CRfcConnection::ConnectParamFromConnectInfo(CRfcString & connect_param)
{
/*
	RFC_MODE is given.  If it is RFC_MODE_R3ONLY, RFC_MODE_VERSION_3,
	nothing special needs to be done.  If it is RFC_MODE_PARAMETER, 
	then it will look into the saprfc.ini file.  If it is RFC_MODE_CPIC_EXT,
	do not use saprfc.ini or sideinfo.  But in order to look into these files,
	DEST must exist.
*/
	if (m_ConnectInfo.rfcMode == RFC_MODE_CPIC
		&& m_ConnectInfo.cRfcServerType == '3')
		m_ConnectInfo.cRfcServerType = '2';

	if (m_ConnectInfo.cRfcServerType != '2' &&
		m_ConnectInfo.cRfcServerType != '3' &&
		m_ConnectInfo.cRfcServerType != 'E' &&
		m_ConnectInfo.cRfcServerType != 'e')
		m_ConnectInfo.cRfcServerType = '3';

	char serverType = m_ConnectInfo.cRfcServerType;

	char tempStr[80]; //  For storing numbers etc in a string
//  ID is TYPE
	if (serverType == '2' || serverType == '3' || serverType == 'E' 
		|| serverType == 'e')
	{
		connect_param += "TYPE=";
		sprintf(tempStr, "%c\0", serverType);
		connect_param += tempStr;
	} 

//  ID is CLIENT, USER, PASSWD, and LANG, used for TYPE = 2 or 3.

	if (serverType == '2' || serverType == '3' || serverType != 'E' || serverType != 'e')
	{
		if (m_UserInfo.rstrClient.GetLength() > 0)
		{
			connect_param += " CLIENT=";
			connect_param += (CSTR)m_UserInfo.rstrClient;
		}
		if (m_UserInfo.rstrUserName.GetLength() > 0)
		{
			connect_param += " USER=";
			connect_param += (CSTR)m_UserInfo.rstrUserName;
		}
		if (m_UserInfo.rstrPassword.GetLength() > 0)
		{
			connect_param += " PASSWD=";
			connect_param += (CSTR)m_UserInfo.rstrPassword;
		}
		if (m_UserInfo.rstrLanguage.GetLength() > 0)
		{
			connect_param += " LANG=";
			connect_param += (CSTR)m_UserInfo.rstrLanguage;
		}
	}
		
//  ID is TRACE, good for all cases

	if (m_nTraceLevel > 99999)  //  There must be an error with the number!
		m_nTraceLevel = 1;
	
	if (m_nTraceLevel > 0)
	{
		connect_param += " TRACE=";
		sprintf(tempStr, "%d\0", m_nTraceLevel);
		connect_param += tempStr;
	}

//  ID is DEST, for all cases, if it is not a NULL

	int lenDest = m_ConnectInfo.rstrDestination.GetLength();

	if (lenDest > 0)
	{
		connect_param += " DEST=";
		connect_param += (CSTR)m_ConnectInfo.rstrDestination;
	}

//  IDs are GWHOST and GWSERV, for R/2 or external systems, if DEST exists,
//  look into saprfc.ini or sideinfo files.

	if ((lenDest <= 0 || m_ConnectInfo.rfcMode == RFC_MODE_CPIC_EXT)
		&& (serverType == '2' || serverType == 'E' || serverType == 'e'))
	{	
		if (m_ConnectInfo.rstrGatewayHost.GetLength() > 0 && m_ConnectInfo.rstrGatewayService.GetLength() > 0)
		{		 
			if (m_ConnectInfo.rstrGatewayHost.GetLength() > 0)
			{
				connect_param += " GWHOST=";
				connect_param += (CSTR)m_ConnectInfo.rstrGatewayHost;
			}
			if (m_ConnectInfo.rstrGatewayService.GetLength() > 0)
			{
				connect_param += " GWSERV=";
				connect_param += (CSTR)m_ConnectInfo.rstrGatewayService;
			}
		}
	} 

//  IDs: MSHOST, GROUP, for all cases

	if (lenDest <= 0 || m_ConnectInfo.rfcMode == RFC_MODE_CPIC_EXT)
	{
		if (m_ConnectInfo.rstrMsgServer.GetLength() > 0)
		{
			connect_param += " MSHOST=";
			connect_param += (CSTR)m_ConnectInfo.rstrMsgServer;
		}
		if (m_ConnectInfo.rstrGroupName.GetLength() > 0)
		{
			connect_param += " GROUP=";
			connect_param += (CSTR)m_ConnectInfo.rstrGroupName;
		}
	}

//  IDs: R3NAME, ASHOST, SYSNR, for R/3 systems only

	if ((lenDest <= 0 || m_ConnectInfo.rfcMode == RFC_MODE_CPIC_EXT)
		&& serverType == '3')
	{
		if (m_ConnectInfo.rstrSystemName.GetLength() > 0)
		{
			connect_param += " R3NAME=";
			connect_param += (CSTR)m_ConnectInfo.rstrSystemName;
		}
		if (m_ConnectInfo.rstrHostName.GetLength() > 0)
		{
			connect_param += " ASHOST=";
			connect_param += (CSTR)m_ConnectInfo.rstrHostName;
		}
		connect_param += " SYSNR=";
		int nSystemNo = m_ConnectInfo.nSystemNo;
		sprintf(tempStr, "%d\0", nSystemNo);
		connect_param += tempStr;
	}

//  IDs: TPHOST, TPNAME, for external systems only

	if ((lenDest <= 0 || m_ConnectInfo.rfcMode == RFC_MODE_CPIC_EXT)
		&& serverType == 'E' || serverType == 'e')
	{
		if (m_ConnectInfo.rstrHostName.GetLength() > 0)
		{
			connect_param += " TPHOST=";
			connect_param += (CSTR)m_ConnectInfo.rstrHostName;
		}
		if (m_ConnectInfo.rstrProgramID.GetLength() > 0)
		{
			connect_param += " TPNAME=";
			connect_param += (CSTR)m_ConnectInfo.rstrProgramID;
		}
	}

	connect_param += " USE_SAPGUI=";
	int nGUI = m_ConnectInfo.nSAPGUI;
	if (nGUI > 99999)  //  There must be an error with the number!
		nGUI = 1;
	else if (nGUI < 0)
		nGUI = 0;
	sprintf(tempStr, "%d\0", nGUI);
	connect_param += tempStr;
//  IDs: ABAP_DEBUG, SNC_MODE, SNC_GOP, SNC_MYNAME, SNC_PARTNERNAME, 
//       SNC_LIB, SAPLOGON_ID, not supported yet.
}

//
//********************************************************************************
//  This method is used to obtain RFC_CONNECT_INFO and RFC_USER_INFO and store them 
//  in the CRfcConnection object from the parameter list in connect_param in for 
//  format used in RfcOpenEx(char * connect_param, RFC_ERROR_INFO_EX * error_info) 
//  Memory is allocated in the calling function for this method.
//*********************************************************************************
//
void CRfcConnection::ConnectInfoFromConnectParam(char * connect_param)
{
	CRfcString paramStr(connect_param);
	int leng = 255;
	char tempStr[256];

//  Search for TYPE
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "TYPE=", leng) == 0 && strlen(tempStr) > 0)
	{
		if (strcmp(tempStr, "2") == 0) 
		{
			m_ConnectInfo.cRfcServerType = '2';
		}
		else if (strcmp(tempStr, "E") == 0 || strcmp(tempStr, "e") == 0)
		{
			m_ConnectInfo.cRfcServerType = 'E';
		}
		else
		{
			m_ConnectInfo.cRfcServerType = '3';
		}
	}
	else
	{
		m_ConnectInfo.cRfcServerType = '3';
	}

//  Search for CLIENT
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "CLIENT=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_UserInfo.rstrClient = tempStr;
	}
	else
		m_UserInfo.rstrClient = NULL;

//  Search for USER
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "USER=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_UserInfo.rstrUserName = tempStr;
	}
	else
		m_UserInfo.rstrUserName = NULL;

//  Search for PASSWD
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "PASSWD=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_UserInfo.rstrPassword = tempStr;
	}
	else
		m_UserInfo.rstrPassword = NULL;
	
//  Search for LANG
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "LANG=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_UserInfo.rstrLanguage = tempStr;
	}
	else
		m_UserInfo.rstrLanguage = NULL;

//	Search for TRACE
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "TRACE=", leng) == 0 && strlen(tempStr) > 0)
	{
		SetTraceLevel(atoi(tempStr));
    }
	else
		SetTraceLevel(0);

//	Search for DEST
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "DEST=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrDestination = tempStr;
	}
	else
		m_ConnectInfo.rstrDestination = NULL;

//	Search for GWHOST
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "GWHOST=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrGatewayHost = tempStr;
	}
	else
	    m_ConnectInfo.rstrGatewayHost = NULL;

//	Search for GWSERV
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "GWSERV=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrGatewayService = tempStr;
	}
	else
	    m_ConnectInfo.rstrGatewayService = NULL;

//  Search for MSHOST
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "MSHOST=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrMsgServer = tempStr;
	}
	else
		m_ConnectInfo.rstrMsgServer = NULL;

//  Search for R3NAME
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "R3NAME=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrSystemName = tempStr;
	}
	else
		m_ConnectInfo.rstrSystemName = NULL;

//  Search for GROUP
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "GROUP=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrGroupName = tempStr;
	}
	else
		m_ConnectInfo.rstrGroupName = NULL;

//  Search for ASHOST
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "ASHOST=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrHostName = tempStr;
	}
	else
		m_ConnectInfo.rstrHostName = NULL;

//  Search for SYSNR
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "SYSNR=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.nSystemNo = atoi(tempStr);
	}
	else
		m_ConnectInfo.nSystemNo = NULL;

//  Search for TPHOST
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "TPHOST=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrHostName = tempStr;
	}

//  Search for TPNAME
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "TPNAME=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.rstrProgramID = tempStr;
	}
	else
		m_ConnectInfo.rstrProgramID = NULL;

//  Search for USE_SAPGUI
	if (paramStr != (CSTR)NULL && paramStr.GetLength() > 0 && paramStr.ParameterSearch(tempStr, "USE_SAPGUI=", leng) == 0 && strlen(tempStr) > 0)
	{
		m_ConnectInfo.nSAPGUI = atoi(tempStr);
	}
	else
		m_ConnectInfo.nSAPGUI = 0;
}
